<?php
namespace Elementor; // Custom widgets must be defined in the Elementor namespace
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly (security measure)

/**
 * Widget Name: Hover Image
 */
class Ronmi_Project_Box extends Widget_Base{

 	// The get_name() method is a simple one, you just need to return a widget name that will be used in the code.
	public function get_name() {
		return 'ot-project-box';
	}

	// The get_title() method, which again, is a very simple one, you need to return the widget title that will be displayed as the widget label.
	public function get_title() {
		return __( 'OT Project Box', 'ronmi' );
	}

	// The get_icon() method, is an optional but recommended method, it lets you set the widget icon. you can use any of the eicon or font-awesome icons, simply return the class name as a string.
	public function get_icon() {
		return 'eicon-image-box';
	}

	// The get_categories method, lets you set the category of the widget, return the category name as a string.
	public function get_categories() {
		return [ 'category_ronmi' ];
	}

	protected function register_controls() {

		/*Content Image*/
		$this->start_controls_section(
			'section_project_content',
			[
				'label' => __( 'Project', 'ronmi' ),
			]
		);

		$this->add_control(
			'image_project',
			[
				'label' => __( 'Choose Image', 'ronmi' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);
		$this->add_control(
			'hover_image_back_icon',
			[
				'label' => __( 'Icon Hover', 'ronmi' ),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default' => [],
				'skin' => 'inline',
				'label_block' => false,
			]
		);
		$this->add_control(
			'title',
			[
				'label' => __( 'Title', 'ronmi' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'Total Control', 'ronmi' ),
				'label_block' => true,
			]
		);
		$this->add_control(
			'title_size',
			[
				'label' => __( 'Title HTML Tag', 'ronmi' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h5',
			]
		);

		$this->add_control(
			'project_cat',
			[
				'label' => __( 'Select Categories', 'ronmi' ),
				'type' => Controls_Manager::SELECT2,
				'options' => $this->select_param_cate_project(),
				'multiple' => true,
				'label_block' => true,
				'placeholder' => __( 'All Categories', 'ronmi' ),
			]
		);

		$this->add_control(
			'link',
			[
				'label' => __( 'Link', 'ronmi' ),
				'type' => Controls_Manager::URL,
				'placeholder' => __( 'https://your-link.com', 'ronmi' ),
			]
		);

		$this->end_controls_section();

		//Style
		$this->start_controls_section(
			'style_section',
			[
				'label' => __( 'Style Content', 'ronmi' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'heading_image',
			[
				'label' => __( 'Image', 'ronmi' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'background_overlay',
			[
				'label' => __( 'Background Overlay', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .ot-project-box .overlay' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label' => __( 'Icon Size', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 6,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .overlay i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .overlay svg' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'icon_color',
			[
				'label' => __( 'Icon Color', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .overlay i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .overlay svg' => 'fill: {{VALUE}};'
				],
			]
		);
		//Title
		$this->add_control(
			'heading_title',
			[
				'label' => __( 'Title', 'ronmi' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		
		$this->add_responsive_control(
			'title_space',
			[
				'label' => __( 'Spacing Bottom', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .ot-project-meta' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => __( 'Color', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .project-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} .project-title a' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'title_color_hover',
			[
				'label' => __( 'Hover', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .project-title a:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'link[url]!' => '',
				]
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .project-title',
			]
		);
		/* category */
		$this->add_control(
			'heading_cat',
			[
				'label' => __( 'Category', 'ronmi' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		$this->add_responsive_control(
			'cat_spacing',
			[
				'label' => __( 'Spacing', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .project-cat' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'cat_color',
			[
				'label' => __( 'Color', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .project-cat a' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'cat_hcolor',
			[
				'label' => __( 'Hover', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .project-cat a:hover' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'cat_typography',
				'selector' => '{{WRAPPER}} .project-cat',
			]
		);

		$this->end_controls_section();

	}

	protected function render() {
		$settings = $this->get_settings_for_display();
		
		$this->add_render_attribute( 'title', 'class', 'project-title' );
		$title_html = sprintf( '<%1$s %2$s>%3$s</%1$s>', $settings['title_size'], $this->get_render_attribute_string( 'title' ), $settings['title'] );

		if ( ! empty( $settings['link']['url'] ) ) {
			$this->add_link_attributes( 'link_box', $settings['link'] );
			$title_html = sprintf( '<%1$s %2$s><a ' .$this->get_render_attribute_string( 'link_box' ). '>%3$s</a></%1$s>', $settings['title_size'], $this->get_render_attribute_string( 'title' ), $settings['title'] );
		}

		$has_icon = ! empty( $settings['icon'] );
		if ( $has_icon ) {
			$this->add_render_attribute( 'i', 'class', $settings['icon'] );
			$this->add_render_attribute( 'i', 'aria-hidden', 'true' );
		}
		if ( ! $has_icon && ! empty( $settings['hover_image_back_icon']['value'] ) ) {
			$has_icon = true;
		}
		$migrated = isset( $settings['__fa4_migrated']['hover_image_back_icon'] );
		$is_new = ! isset( $settings['icon'] ) && Icons_Manager::is_migration_allowed();

		$cats = $settings['project_cat'];
		?>

		<div class="ot-project-box">
			<div class="ot-project-meta">
				<?php if( !empty( $cats )){ ?>
					<div class="project-cat">
						<?php 
							foreach( (array)$cats as $cat){
	                        $cates = get_term_by('term_id', $cat, 'portfolio_cat');
	                        $cat_name = $cates->name;
	                        $cat_id   = $cates->term_id;
	                        $category_link = get_category_link( $cat_id );
						?>
						<span>
							<a href="<?php echo esc_url($category_link) ?>"><?php echo esc_html( $cat_name );?></a>
						</span>
						<?php } ?> 
					</div>
				<?php } ?>
				<?php if( !empty( $settings['title'] ) ) { echo $title_html; } ?>
			</div>
			<figure class="ot-project-image">
				<?php echo Group_Control_Image_Size::get_attachment_image_html( $settings, 'image_project' ); ?>
				<?php if ( $has_icon ) : ?>
				<span class="overlay flex-middle">
					<?php
						if ( $is_new || $migrated ) {
							Icons_Manager::render_icon( $settings['hover_image_back_icon'], [ 'aria-hidden' => 'true' ] );
						} elseif ( ! empty( $settings['icon'] ) ) {
							?><i <?php $this->print_render_attribute_string( 'i' ); ?>></i><?php
						}
					?>
				</span>
				<?php endif; ?>
			</figure>
	    </div>

	    <?php
	}

	protected function select_param_cate_project() {
		$category = get_terms( 'portfolio_cat' );
		$cat = array();
		foreach( $category as $item ) {
		    if( $item ) {
			  	$cat[$item->term_id] = $item->name;
		   	}
		}
		return $cat;
	}

}
// After the Ronmi_Project_Box class is defined, I must register the new widget class with Elementor:
Plugin::instance()->widgets_manager->register_widget_type( new Ronmi_Project_Box() );