<?php 
namespace Elementor; // Custom widgets must be defined in the Elementor namespace
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly (security measure)

/**
 * Widget Name: Client Logos
 */
class Ronmi_Client_Logo extends Widget_Base{

 	// The get_name() method is a simple one, you just need to return a widget name that will be used in the code.
	public function get_name() {
		return 'ot-client-logos';
	}

	// The get_title() method, which again, is a very simple one, you need to return the widget title that will be displayed as the widget label.
	public function get_title() {
		return __( 'OT Client Logo', 'ronmi' );
	}

	// The get_icon() method, is an optional but recommended method, it lets you set the widget icon. you can use any of the eicon or font-awesome icons, simply return the class name as a string.
	public function get_icon() {
		return 'eicon-slider-push';
	}

	// The get_categories method, lets you set the category of the widget, return the category name as a string.
	public function get_categories() {
		return [ 'category_ronmi' ];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Logo', 'ronmi' ),
			]
		);

		$repeater = new Repeater();
		$repeater->add_control(
			'title',
			[
				'label' => __( 'Name', 'ronmi' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( '', 'ronmi' ),
			]
		);
		$repeater->add_control(
			'type_logo',
			[
				'label' => __( 'Type Logo', 'ronmi' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'image',
				'options' => [
					'image' => [
						'title' => __( 'Image', 'ronmi' ),
						'icon' => 'eicon-image',
					],
					'icon' => [
						'title' => __( 'Icon', 'ronmi' ),
						'icon' => 'eicon-star',
					],
				],
				'toggle' => false,
			]
		);
		$repeater->add_control(
			'image_partner',
			[
				'label' => __( 'Image', 'ronmi' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'type_logo' => 'image'
				]
			]
		);
		$repeater->add_control(
			'image_partner_hover',
			[
				'label' => __( 'Image Hover', 'ronmi' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'type_logo' => 'image'
				]
			]
		);
		$repeater->add_control(
			'icon_partner',
			[
				'label' => __( 'Icon', 'ronmi' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-star',
					'library' => 'fa-solid',
				],
				'condition' => [
					'type_logo' => 'icon'
				]
			]
		);
		$repeater->add_control(
			'icon_partner_hover',
			[
				'label' => __( 'Icon Hover', 'ronmi' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-star',
					'library' => 'fa-solid',
				],
				'condition' => [
					'type_logo' => 'icon'
				],
			]
		);
		$repeater->add_control(
			'logo_link',
			[
				'label' => __( 'Link', 'ronmi' ),
				'type' => Controls_Manager::URL,
				'default' => [
					'url' => '#',
				],
			]
		);
		$this->add_control(
		    'image_carousel',
		    [
		        'label'       => '',
		        'type'        => Controls_Manager::REPEATER,
		        'show_label'  => false,
		        'default'     => [],
		        'fields'      => $repeater->get_controls(),
		        'title_field' => '{{{title}}}',
		    ]
		);

		$slides_show = range( 1, 10 );
		$slides_show = array_combine( $slides_show, $slides_show );

		$this->add_responsive_control(
			'tshow',
			[
				'label' => __( 'Slides To Show', 'ronmi' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'' => __( 'Default', 'ronmi' ),
				] + $slides_show,
				'default' => ''
			]
		);

		$this->add_control(
			'loop',
			[
				'label'   => esc_html__( 'Loop', 'ronmi' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);
		$this->add_control(
			'autoplay',
			[
				'label'   => esc_html__( 'Autoplay', 'ronmi' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);
		$this->add_control(
			'timeout',
			[
				'label' => __( 'Autoplay Timeout', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min'  => 1000,
						'max'  => 20000,
						'step' => 1000,
					],
				],
				'default' => [
					'size' => 7000,
				],
				'condition' => [
					'autoplay' => 'yes',
				]
			]
		);
		
		$this->add_control(
			'navigation',
			[
				'label' => __( 'Navigation', 'ronmi' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'none',
				'options' => [
					'both' => __( 'Arrows and Dots', 'ronmi' ),
					'arrows' => __( 'Arrows', 'ronmi' ),
					'dots' => __( 'Dots', 'ronmi' ),
					'none' => __( 'None', 'ronmi' ),
				],
			]
		);

		$this->end_controls_section();

		//Style

		$this->start_controls_section(
			'image_style_section',
			[
				'label' => __( 'Image', 'ronmi' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'logo_vertical_align',
			[
				'label' => __( 'Vertical Align', 'ronmi' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'flex-start' => [
						'title' => __( 'Start', 'ronmi' ),
						'icon' => 'eicon-v-align-top',
					],
					'center' => [
						'title' => __( 'Center', 'ronmi' ),
						'icon' => 'eicon-v-align-middle',
					],
					'flex-end' => [
						'title' => __( 'End', 'ronmi' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'condition' => [
					'tshow!' => '1',
				],
				'selectors' => [
					'{{WRAPPER}} .owl-stage' => 'display: flex; align-items: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'img_width',
			[
				'label' => __( 'Width', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .owl-item img' => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .ot-client-logo-slider .owl-item svg' => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .ot-client-logo-slider .owl-item i:before' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'img_spacing',
			[
				'label' => __( 'Spacing', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 80,
				],
			]
		);
		$this->add_responsive_control(
			'logo_padding',
			[
				'label' => 'Padding',
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .client_link span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'logo_radius',
			[
				'label' => __( 'Border Radius', 'ronmi' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .client_link span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'img_effects' );

		$this->start_controls_tab( 'normal',
			[
				'label' => __( 'Normal', 'ronmi' ),
			]
		);

		$this->add_control(
			'svg_color',
			[
				'label' => __( 'Color ( For Icon SVG )', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '#696871',
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .owl-item svg' => 'fill: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'bg_color',
			[
				'label' => __( 'Background', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .client-logo' => 'background: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'opacity',
			[
				'label' => __( 'Opacity', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .client-logo img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'img_css_filters',
				'selector' => '{{WRAPPER}} .ot-client-logo-slider .client-logo img',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'img_hover_effects',
			[
				'label' => __( 'Hover', 'ronmi' ),
			]
		);

		$this->add_control(
			'svg_hcolor',
			[
				'label' => __( 'Color ( For Icon SVG )', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '#696871',
				'selectors' => [
					'{{WRAPPER}} .client_link .client-hover-logo svg' => 'fill: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'bg_hcolor',
			[
				'label' => __( 'Background', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .client-hover-logo' => 'background: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'opacity_hover',
			[
				'label' => __( 'Opacity', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .ot-client-logo-slider .client-hover-logo img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'img_css_filters_hover',
				'selector' => '{{WRAPPER}} .ot-client-logo-slider .client-hover-logo img',
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		// Dots.
		$this->start_controls_section(
			'navigation_section',
			[
				'label' => __( 'Dots', 'ronmi' ),
				'tab' => Controls_Manager::TAB_STYLE,
				'condition' => [
					'navigation' => [ 'dots', 'both' ],
				],
			]
		);

		$this->add_control(
			'dots_style',
			[
				'label'        => esc_html__( 'Dots Style', 'ronmi' ),
				'type'         => Controls_Manager::SELECT,
				'default' => 'classic',
				'options' => [
					'classic' => __( 'Classic', 'ronmi' ),
					'custom' => __( 'Custom', 'ronmi' ),
				],
			]
		);

		$this->add_responsive_control(
			'dots_spacing',
			[
				'label' => __( 'Spacing', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .owl-dots' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
            'dots_bgcolor',
            [
                'label' => __( 'Color', 'ronmi' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
					'{{WRAPPER}} .owl-dots .owl-dot span' => 'background: {{VALUE}};',
					'{{WRAPPER}} .owl-dots .owl-dot:hover, {{WRAPPER}} .owl-dots .owl-dot.active' => 'border-color: {{VALUE}};'
				],
            ]
        );

        $this->add_control(
            'dots_active_bgcolor',
            [
                'label' => __( 'Color Active', 'ronmi' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
					'{{WRAPPER}} .owl-dots .owl-dot:hover span, {{WRAPPER}} .owl-dots .owl-dot.active span' => 'background: {{VALUE}};',
					'{{WRAPPER}} .owl-dots .owl-dot:hover, {{WRAPPER}} .owl-dots .owl-dot.active' => 'border-color: {{VALUE}};'
				],
            ]
        );

        $this->end_controls_section();

        // Arrow.
		$this->start_controls_section(
			'style_nav',
			[
				'label' => __( 'Arrows', 'ronmi' ),
				'tab' => Controls_Manager::TAB_STYLE,
				'condition' => [
					'navigation' => [ 'arrows', 'both' ],
				]
			]
		);
		$this->add_responsive_control(
			'arrow_spacing',
			[
				'label' => __( 'Spacing', 'ronmi' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .owl-nav button.owl-prev' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .owl-nav button.owl-next' => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'arrow_color',
			[
				'label' => __( 'Color', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .owl-nav button' => 'color: {{VALUE}};',
				]
			]
		);
		$this->add_control(
			'arrow_hcolor',
			[
				'label' => __( 'Color Hover', 'ronmi' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .owl-nav button:hover' => 'color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();

	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['image_carousel'] ) ) {
			return;
		}
		$shows  = !empty( $settings['tshow'] ) ? $settings['tshow'] : 3;
		$tshows = !empty( $settings['tshow_tablet'] ) ? $settings['tshow_tablet'] : $shows;
		$mshows = !empty( $settings['tshow_mobile'] ) ? $settings['tshow_mobile'] : $tshows;
		$gaps   = isset( $settings['img_spacing']['size'] ) && is_numeric( $settings['img_spacing']['size'] ) ? $settings['img_spacing']['size'] : 80;
		$tgaps  = isset( $settings['img_spacing_tablet']['size'] ) && is_numeric( $settings['img_spacing_tablet']['size'] ) ? $settings['img_spacing_tablet']['size'] : $gaps;
		$mgaps  = isset( $settings['img_spacing_mobile']['size'] ) && is_numeric( $settings['img_spacing_mobile']['size'] ) ? $settings['img_spacing_mobile']['size'] : $tgaps;
		$timeout  = isset( $settings['timeout']['size'] ) ? $settings['timeout']['size'] : '';
		$dots   = ( in_array( $settings['navigation'], [ 'dots', 'both' ] ) );
		$arrows = ( in_array( $settings['navigation'], [ 'arrows', 'both' ] ) );
		$owl_options = [
			'slides_show_desktop'  => absint( $shows ),
			'slides_show_tablet'   => absint( $tshows ),
			'slides_show_mobile'   => absint( $mshows ),
			'margin_desktop'   	   => absint( $gaps ),
			'margin_tablet'   	   => absint( $tgaps ),
			'margin_mobile'   	   => absint( $mgaps ),
			'autoplay'      	   => $settings['autoplay'] ? $settings['autoplay'] : 'no',
			'autoplay_time_out'    => absint( $timeout ),
			'loop'      		   => $settings['loop'] ? $settings['loop'] : 'no' ,
			'arrows'        	   => $arrows,
			'dots'          	   => $dots,
			'dots_style'		   => $settings['dots_style']
		];

		$this->add_render_attribute(
			'slides', [
				'class'               => 'ot-slider ot-client-logo-slider',
				'data-slider_options' => wp_json_encode( $owl_options ),
			]
		);

		?>
		<div <?php echo $this->get_render_attribute_string( 'slides' ); ?>>
			<div class="owl-carousel owl-theme">

			<?php
			foreach ( $settings['image_carousel'] as $key => $attachment ) {
				$title = $attachment['title'];
				$slide_html= '';
				
				$this->add_render_attribute( 'link' . $key, 'class', 'client_link'  );
				$this->add_render_attribute( 'link' . $key, 'href', $attachment['logo_link']['url'] ? $attachment['logo_link']['url'] : '#' );

				if ( $attachment['logo_link']['is_external'] ) {
					$this->add_render_attribute( 'link' . $key, 'target', '_blank' );
				}

				if ( $attachment['logo_link']['nofollow'] ) {
					$this->add_render_attribute( 'link' . $key, 'rel', 'nofollow' );
				}
	            
				if( $attachment['type_logo'] == 'image' ){
					$logo_html = '<span class="client-logo"><img src="' . esc_attr( $attachment['image_partner']['url'] ) . '" alt="' . esc_attr( $title ) . '"></span>';
	            	$logo_hover_html = '<span class="client-hover-logo"><img src="' . esc_attr( $attachment['image_partner_hover']['url'] ) . '" alt="' . esc_attr( $title ) . '"></span>';
	            	$slide_html = '<a '.$this->get_render_attribute_string('link' . $key). '>' . $logo_html . $logo_hover_html .'</a>';
	            	echo $slide_html;
				}else{
					?>
					<a <?php echo $this->get_render_attribute_string( 'link' . $key ); ?> >
						<span class="client-logo">
							<?php Icons_Manager::render_icon( $attachment['icon_partner'], [ 'aria-hidden' => 'true' ] ); ?>
						</span>
						<span class="client-hover-logo">
							<?php Icons_Manager::render_icon( $attachment['icon_partner_hover'], [ 'aria-hidden' => 'true' ] ); ?>
						</span>
					</a>
					<?php
				}
			}
			?>

			</div>
	    </div>
		<?php 
		
	}

	protected function content_template() {}

	public function get_keywords() {
		return [ 'slider', 'carousel', 'logo', 'client' ];
	}

}
// After the Ronmi_Client_Logo class is defined, I must register the new widget class with Elementor:
Plugin::instance()->widgets_manager->register_widget_type( new Ronmi_Client_Logo() );